// This is an adaptation of Minimig top hierarchy source file for Pong Game from fpga4fun.com
// Visit http://www.fpga4fun.com/PongGame.html for details and necessary source files
//
// The paddle is controlled by an Amiga mouse connected to the first joystick port.
// The value of pixel clock is calculated to output video signal very close to 60Hz 640x480 VGA resolution.
//
// This file was created to demonstrate use of DCM module.
//
// Author: Jakub Bednarski
//
// Minimig is copyrighted 2006-2009 by Dennis van Weeren 
//
// Minimig is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//

module minipong
(
	//m68k pins
	inout 	[15:0]cpudata,		//m68k data bus
	input	[23:1]cpuaddress,	//m68k address bus
	output	[2:0]_ipl,			//m68k interrupt request
	input	_as,				//m68k address strobe
	input	_uds,				//m68k upper data strobe
	input	_lds,				//m68k lower data strobe
	input	r_w,				//m68k read / write
	output	_dtack,				//m68k data acknowledge
	output	_cpureset,			//m68k reset
	output	cpuclk,				//m68k clock
	//sram pins
	inout	[15:0]ramdata,		//sram data bus
	output	[19:1]ramaddress,	//sram address bus
	output	_ramsel0,			//sram enable bank 0
	output	_ramsel1,			//sram enable bank 1
	output	_ub,				//sram upper byte select
	output	_lb,				//sram lower byte select
	output	_we,				//sram write enable
	output	_oe,				//sram output enable
	//system	pins
	input	mclk,				//master system clock (4.433619MHz)
	//rs232 pins
	input	rxd,				//rs232 receive
	output	txd,				//rs232 send
	input	cts,				//rs232 clear to send
	output	rts,				//rs232 request to send
	//I/O
	input	[5:0]_joy1,			//joystick 1 [fire2,fire,up,down,left,right] (default mouse port)
	input	[5:0]_joy2,			//joystick 2 [fire2,fire,up,down,left,right] (default joystick port)
	input	_15khz,				//scandoubler disable
	output	pwrled,				//power led
	inout	msdat,				//PS2 mouse data
	inout	msclk,				//PS2 mouse clk
	inout	kbddat,				//PS2 keyboard data
	inout	kbdclk,				//PS2 keyboard clk
	//host controller interface (SPI)
	input	_spisel0,			//SPI enable 0
	input	_spisel1,			//SPI enable 1
	input	_spisel2,			//SPI enable 2
	input	spidin,				//SPI data input
	inout	spidout,			//SPI data output
	input	spiclk,				//SPI clock
	//video
	output	_hsyncout,			//horizontal sync
	output	_vsyncout,			//vertical sync
	output	[3:0]redout,		//red
	output	[3:0]greenout,		//green
	output	[3:0]blueout,		//blue
	//audio
	output	left,				//audio bitstream left
	output	right,				//audio bitstream right
	//user i/o
	output	gpio0,
	output	gpio1,
	output	gpio2
);

//--------------------------------------------------------------------------------------

	wire	clk;
	wire	vga_R;
	wire	vga_G;
	wire	vga_B;
	
//--------------------------------------------------------------------------------------

	//assign unused outputs to inactive safe states
	assign	pwrled = 1'b1;
	assign	_ipl[2:0] = 3'b111;
	assign	_dtack = 1;
	assign	_cpureset = 0;
	assign	cpuclk = 0;
	assign	ramdata[15:0] = 16'bzzzz_zzzz_zzzz_zzzz;
	assign	ramaddress[19:1] = 0;
	assign	_ramsel0 = 1;
	assign	_ramsel1 = 1;
	assign	_ub = 1;
	assign	_lb = 1;
	assign	_we = 1;
	assign	_oe = 1;
	assign	txd = 1;
	assign	rts = 1;
	assign	left = 0;
	assign	right = 0;
	assign	msdat = 1'bz;
	assign	msclk = 1'bz;
	assign	kbddat = 1'bz;
	assign	kbdclk = 1'bz;
	assign	spidout = 1'bz;
	assign	gpio2 = 1'bz;
	assign	gpio1 = 1'bz;
	assign	gpio0 = 1'bz;

//--------------------------------------------------------------------------------------

//instantiate clock generator
clkgen CG1
(	
	.clkin(mclk),
	.clkout(clk25)
);

//instantiate pong (get sources from http://www.fpga4fun.com/PongGame.html)
pong P1
(
	.clk25(clk25), 
	.hsync(_hsyncout), 
	.vsync(_vsyncout), 
	.rota(_joy1[0]), 
	.rotb(_joy1[2]),
	.green(greenout),
	.blue(blueout),
	.red(redout)
);

//assign VGA colour outputs
//assign	redout[3:0] = {4{vga_R}};
//assign	greenout[3:0] = {4{vga_G}};
//assign	blueout[3:0] = {4{vga_B}};
	
endmodule

module clkgen
(
	input	clkin,		// 4.433619 MHz input clock
	output	clkout	 	// 25.123841 (=clkin*17/3) output clock output
);


// To generate 60 Hz vertical sync VGA signal we need a pixel clock of 768*512*60 = 23.592960 MHz
// If we multiply input clock of 4.433619 MHz by 16/3 we get 23.645968 MHz which is pretty close to the desired value (0.3% mismatch)

	wire	dcm_clkin;
	wire	dcm_clkout;
   
	//clock input buffer
	IBUFG clkin_buf ( .I(clkin), .O(dcm_clkin) );	
	
	DCM #
	(	
		.CLKDV_DIVIDE(3.0),						// Divide by: 1.5,2.0,2.5,3.0,3.5,4.0,4.5,5.0,5.5,6.0,6.5,7.0,7.5,8.0,9.0,10.0,11.0,12.0,13.0,14.0,15.0 or 16.0
		.CLKFX_DIVIDE(3),						// Can be any integer from 1 to 32
		.CLKFX_MULTIPLY(17),					// Can be any integer from 2 to 32
		.CLKIN_DIVIDE_BY_2("FALSE"),			// TRUE/FALSE to enable CLKIN divide by two feature
		.CLKIN_PERIOD(225.0),					// Specify period of input clock
		.CLKOUT_PHASE_SHIFT("NONE"),			// Specify phase shift of NONE, FIXED or VARIABLE
		.CLK_FEEDBACK("NONE"), 					// Specify clock feedback of NONE, 1X or 2X
		.DESKEW_ADJUST("SYSTEM_SYNCHRONOUS"),	// SOURCE_SYNCHRONOUS, SYSTEM_SYNCHRONOUS or an integer from 0 to 15
		.DFS_FREQUENCY_MODE("LOW"), 			// HIGH or LOW frequency mode for frequency synthesis
		.DLL_FREQUENCY_MODE("LOW"),				// HIGH or LOW frequency mode for DLL
		.DUTY_CYCLE_CORRECTION("TRUE"),			// Duty cycle correction, TRUE or FALSE
		.FACTORY_JF(16'hC080),					// FACTORY JF values
		.PHASE_SHIFT(0),						// Amount of fixed phase shift from -255 to 255
		.STARTUP_WAIT("FALSE")					// Delay configuration DONE until DCM LOCK, TRUE/FALSE
	)
	DCM1
	(
		.CLKFX(dcm_clkout),		// DCM CLK synthesis out (M/D) (allowed range 18-210 MHz)
		.CLKIN(dcm_clkin)		// Clock input (from IBUFG, BUFG or DCM)
	);

	//global clock buffer
	BUFG clkout_buf ( .I(dcm_clkout), .O(clkout) );

endmodule
